import { world } from "@minecraft/server";
import { OVERWORLD, getHostPlayer } from "./main";
import { getClosestEntityFromLocation, getClosestPlayerFromLocation, vector3ToString } from "./utils";
import { ActionFormData } from "@minecraft/server-ui";
const ENTITY_ID_TESLA_GATE = "lc:scpd_tesla_gate";
const ENTITY_ID_TESLA_GATE_TOGGLE_SIGNAL = "lc:scpd_tesla_gate_toggle_signal"; // Tesla Gate toggle signal will toggle on and off a tesla gate entity
const ENTITY_ID_TESLA_GATE_MANAGER_EVNTSIG = "lc:scpd_tesla_gate_manager_evntsig"; // On interact tesla gate manager
const EQO_TESLA_GATE = {
    type: ENTITY_ID_TESLA_GATE,
};
class TeslaGateManagerFormWrapper {
    constructor(teslaGateEntity, form) {
        this._teslaGateEntities = teslaGateEntity;
        this._form = form;
    }
    get teslaGateEntities() {
        return this._teslaGateEntities;
    }
    get form() {
        return this._form;
    }
}
function runTeslaGateEntityCheckAtLocation(location) {
    return OVERWORLD.runCommand(`testfor @e[type=${ENTITY_ID_TESLA_GATE},r=0.8,x=${location.x},y=${location.y},z=${location.z}]`);
}
function generateTeslaGateManagerFormWrapper() {
    let form = new ActionFormData();
    let teslaGateEntities = OVERWORLD.getEntities(EQO_TESLA_GATE);
    teslaGateEntities.sort((a, b) => a.nameTag.localeCompare(b.nameTag));
    form.title("Nearby Tesla Gates");
    form.body(`${teslaGateEntities.length} Tesla Gates found.\nSelect a Tesla Gate to toggle.\n§2Green: INACTIVE§f, §4Red: ACTIVE§f`);
    if (teslaGateEntities.length === 0) {
        form.button("NO TESLA GATES FOUND!");
    }
    for (let i = 0; i < teslaGateEntities.length; i++) {
        const teslaGateEntity = teslaGateEntities[i];
        let teslaGateLocation = teslaGateEntity.location;
        let teslaGateState = teslaGateEntity.getProperty("tesla_gate:state");
        let buttonContent;
        if (teslaGateEntity.nameTag.length > 0) {
            buttonContent = teslaGateEntity.nameTag;
        }
        else {
            buttonContent = `X: ${Math.floor(teslaGateLocation.x)}, Y: ${Math.floor(teslaGateLocation.y)}, Z: ${Math.floor(teslaGateLocation.z)}`;
        }
        buttonContent = (teslaGateState === 0 ? "§2" : "§4") + buttonContent;
        form.button(buttonContent);
    }
    let formWrapper = new TeslaGateManagerFormWrapper(teslaGateEntities, form);
    return formWrapper;
}
function onTeslaGateToggleSignalSpawned(signalEntity) {
    let signalEntityLocation = signalEntity.location;
    if (runTeslaGateEntityCheckAtLocation(signalEntityLocation).successCount <= 0) {
        world.sendMessage(`§cteslaGate.js: No tesla gate entity found at ${vector3ToString(signalEntityLocation)}`);
        signalEntity.triggerEvent("despawn");
        return;
    }
    let teslaGateEntity = getClosestEntityFromLocation(EQO_TESLA_GATE, OVERWORLD, signalEntityLocation);
    if (teslaGateEntity === undefined) {
        world.sendMessage(`§cteslaGate.js: No tesla gate entity found at ${vector3ToString(signalEntityLocation)}`);
        signalEntity.triggerEvent("despawn");
        return;
    }
    teslaGateEntity.triggerEvent("tesla_gate:toggle");
    signalEntity.triggerEvent("despawn");
}
function onTeslaGateManagerEventSignalSpawned(signalEntity) {
    let formWrapper = generateTeslaGateManagerFormWrapper();
    let playerToSeeForm = getClosestPlayerFromLocation(getHostPlayer(), signalEntity.location);
    signalEntity.triggerEvent("despawn");
    formWrapper.form.show(playerToSeeForm).then(response => {
        if (formWrapper.teslaGateEntities.length === 0) {
            return;
        }
        if (response.canceled) {
            return;
        }
        let selectionIndex = response.selection; // Index of tesla gate
        let teslaGateEntity = formWrapper.teslaGateEntities[selectionIndex];
        teslaGateEntity.triggerEvent("tesla_gate:toggle"); // Toggle ON or OFF tesla gate
        let msg;
        let onOffText;
        if (teslaGateEntity.getProperty("tesla_gate:state") === 0) {
            onOffText = "§4ACTIVE";
        }
        else {
            onOffText = "§2INACTIVE";
        } // Property change in tesla_gate:toggle will not apply immediately so check condition with previous value
        if (teslaGateEntity.nameTag.length <= 0) {
            msg = `§aToggled Tesla Gate at ${vector3ToString(teslaGateEntity.location)} to §l${onOffText}§r`;
        }
        else {
            msg = `§aToggled Tesla Gate §o"${teslaGateEntity.nameTag}"§r§a to §l${onOffText}§r`;
        }
        playerToSeeForm.sendMessage(msg);
    });
}
export function setupTeslaGateSystem() {
    world.afterEvents.entitySpawn.subscribe(event => {
        let spawnedEntity = event.entity;
        switch (spawnedEntity.typeId) {
            case ENTITY_ID_TESLA_GATE_TOGGLE_SIGNAL:
                onTeslaGateToggleSignalSpawned(spawnedEntity);
                break;
            case ENTITY_ID_TESLA_GATE_MANAGER_EVNTSIG:
                onTeslaGateManagerEventSignalSpawned(spawnedEntity);
                break;
            default:
                break;
        }
    });
}
